<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Carbon;

class PerformanceController extends Controller
{
    public function index()
    {
        if (! auth()->user()->can('business_settings.access')) {
            abort(403, 'Unauthorized action.');
        }

        $stats = [];
        $since = now()->subDay();
        $qry = DB::table('slow_queries')->where('created_at', '>=', $since);
        $stats['slow_24h_count'] = (clone $qry)->count();
        $stats['slow_24h_avg_ms'] = (clone $qry)->avg('time_ms') ?: 0;
        $stats['slow_24h_max_ms'] = (clone $qry)->max('time_ms') ?: 0;
        $top = (clone $qry)->orderByDesc('time_ms')->limit(10)->get();

        // Environment info
        $env = [
            'php' => PHP_VERSION,
            'laravel' => app()->version(),
            'cache' => config('cache.default'),
            'queue' => config('queue.default'),
            'opcache_enabled' => function_exists('opcache_get_status') ? (bool) (opcache_get_status(false)['opcache_enabled'] ?? false) : false,
        ];

        // Check expected index presence (MySQL only)
        $index_names = [
            'transactions' => ['trx_biz_type_status_idx','trx_biz_type_loc_idx','trx_biz_type_tdate_idx','trx_biz_type_ddate_idx','trx_approval_status_idx','trx_created_by_idx'],
            'purchase_lines' => ['pl_transaction_id_idx','pl_product_id_idx','pl_variation_id_idx','pl_pr_line_id_idx'],
            'variation_location_details' => ['vld_variation_location_idx'],
            'products' => ['prod_biz_stock_active_idx','prod_alert_qty_idx','prod_category_id_idx','prod_brand_id_idx','prod_sku_idx','prod_name_idx'],
            'transaction_sell_lines' => ['tsl_transaction_id_idx','tsl_product_id_idx','tsl_variation_id_idx','tsl_parent_line_id_idx'],
            'transaction_payments' => ['tp_transaction_id_idx','tp_payment_ref_no_idx','tp_method_idx'],
        ];
        $db = DB::getDatabaseName();
        $indexes = [];
        try {
            foreach ($index_names as $table => $names) {
                $existing = DB::select('SELECT INDEX_NAME as name FROM information_schema.STATISTICS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?', [$db, $table]);
                $set = collect($existing)->pluck('name')->all();
                $indexes[$table] = array_map(function ($n) use ($set) { return [ 'name' => $n, 'present' => in_array($n, $set, true)]; }, $names);
            }
        } catch (\Throwable $e) {
            $indexes = [];
        }

        return view('performance.index', compact('stats','top','env','indexes'));
    }

    public function download()
    {
        if (! auth()->user()->can('business_settings.access')) {
            abort(403, 'Unauthorized action.');
        }

        $since = now()->subDay();
        $rows = DB::table('slow_queries')
            ->where('created_at', '>=', $since)
            ->orderByDesc('time_ms')
            ->get();

        $filename = 'slow-queries-' . now()->format('Ymd_His') . '.csv';

        return response()->streamDownload(function () use ($rows) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['time_ms', 'sql', 'url', 'created_at']);

            foreach ($rows as $row) {
                $createdAt = $row->created_at
                    ? Carbon::parse($row->created_at)->timezone(config('app.timezone'))->format('Y-m-d H:i:s')
                    : '';

                $sql = preg_replace('/\s+/', ' ', (string) $row->sql);

                fputcsv($handle, [
                    (int) $row->time_ms,
                    $sql,
                    $row->url,
                    $createdAt,
                ]);
            }

            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
        ]);
    }

    public function clean()
    {
        if (! auth()->user()->can('business_settings.access')) {
            abort(403, 'Unauthorized action.');
        }

        $purge = (bool) request()->input('purge_slow', false);
        $keepDays = (int) request()->input('keep_days', 7);

        try {
            // Clear caches and rebuild config cache
            Artisan::call('optimize:clear');
            Artisan::call('cache:clear');
            Artisan::call('view:clear');
            Artisan::call('config:cache');
            // Reset Spatie permission cache if installed
            try { Artisan::call('permission:cache-reset'); } catch (\Throwable $e) {}
            // Gently restart queue workers if any
            try { Artisan::call('queue:restart'); } catch (\Throwable $e) {}

            if ($purge) {
                $cutoff = now()->subDays(max(0, $keepDays));
                DB::table('slow_queries')->where('created_at', '<', $cutoff)->delete();
            }

            return back()->with('status', [
                'success' => 1,
                'msg' => 'Caches cleared successfully'.($purge ? ' and old slow query logs purged.' : '.'),
            ]);
        } catch (\Throwable $e) {
            return back()->with('status', [
                'success' => 0,
                'msg' => $e->getMessage(),
            ]);
        }
    }
}
